#include <hidef.h>
#include "ts_functions.h" 
#include "comm.h"
#include "control_constants.h"


word xdisplay;
word ydisplay;
word xmeas;
word ymeas;

word xmax;
word ymax;
word xmin;
word ymin;

extern char num_readings;

word Aprime;
word Bprime;
word Cprime;
word Dprime;
word Eprime;
word Fprime;
word xtemp;
word ytemp;
word Atemp;
word Btemp;
word Dtemp;
word Etemp;
word *cal_ptr;



/**************************************************************
*    Function:     Get_Point(mode,num)
*    Parameters: mode - what touchscreen parameter is being measureed
*                num  - the number of a/d readings that should be measured
*                       and used to take a median value
*    Return: the touchscreen value (either touched coordinate or min/max value)
*
*    This function will take 'num' continuous a/d readings and then
*      call the median function which will pick the median a/d reading
*      This median value is sent back to the calling routine. 
*      The A/D is configured to run in continuous mode when turned-on
*      The A/D is turned ON by setting AtoD_Convert to the channel
*      you want to read.
***************************************************************/

word Get_Point (char mode, char num)
{  
    byte x;
    char mid_index = (char)(num+1)>>1;          //The middle index of the sampled array
    static word AtoD_readings[MAX_MEDIAN_LEN];  // make static so that it is not on stack
  
    switch(mode)
    {     
        case read_xcoor:                //Take a x-coordinate reading
            //Drive the x-axis and float the y-axis
            XP = Output;              
            XN = Output;      
            YP = Input;
            YN = Input;
            XP_drive = VDD;
            XN_drive = GND;
            DelayUsec(sample_delay_usec);  //Wait a few us to let the voltages stabailize    
            AtoD_Convert = YP_SNS;         //Start the A/D on the YP sense pin
            break;
      
        case read_ycoor:                //Take a y-coordinate reading
            //Drive the y-axis and float the x-axis.
            XP = Input;              
            XN = Input;      
            YP = Output;
            YN = Output;
            YP_drive = VDD;
            YN_drive = GND;      
            DelayUsec(sample_delay_usec); //Wait a few us to let the voltages stabailize     
            AtoD_Convert = XP_SNS;        //Start the A/D on the XP sense pin
            break;
    
        case read_xmax:                 //Calibrate the touchscreen's X-positive pad
            //Drive the x-axis and float the y-axis
            XP = Output;              
            XN = Output;      
            YP = Input;
            YN = Input;
            XP_drive = VDD;
            XN_drive = GND;      
            DelayUsec(sample_delay_usec); //Wait a few us to let the voltages stabailize     
            AtoD_Convert = XP_SNS;        //Start the A/D on the XP sense pin      
            break;

        case read_xmin:                 //Calibrate the touchscreen's X-negative pad
            //Drive the x-axis and float the y-axis
            XP = Output;              
            XN = Output;      
            YP = Input;
            YN = Input;
            XP_drive = VDD;
            XN_drive = GND;      
            DelayUsec(sample_delay_usec); //Wait a few us to let the voltages stabailize     
            AtoD_Convert = XN_SNS;        //Start the A/D on the XN sense pin      
            break;

        case read_ymax:                 //Calibrate the touchscreen's Y-positive pad
            //Drive the y-axis and float the x-axis
            XP = Input;              
            XN = Input;      
            YP = Output;
            YN = Output;
            YP_drive = VDD;
            YN_drive = GND;      
            DelayUsec(sample_delay_usec); //Wait a few us to let the voltages stabailize     
            AtoD_Convert = YP_SNS;        //Start the A/D on the YP sense pin      
            break;
    
        case read_ymin:                 //Calibrate the touchscreen's Y-negative pad      
            //Drive the y-axis and float the x-axis
            XP = Input;              
            XN = Input;      
            YP = Output;
            YN = Output;
            YP_drive = VDD;
            YN_drive = GND;                  
            DelayUsec(sample_delay_usec); //Wait a few us to let the voltages stabailize     
            AtoD_Convert = YN_SNS;        //Start the A/D on the YN sense pin      
            break;
    
        default:
            return 0;                   
    }

//..... Take "num" A/D readings and store in a temporary array.  

    for (x=0; x<num; x++)               //Take continuous readings
    {        
        while(!ADCSC1_COCO);            //Wait until conversion is complete: CoCo is the conversion compelte flag
        AtoD_readings[x] = ADCR;        //Read the A/D value: ADCR is the 16-bit result register
    }

    AtoD_Convert = AtoD_Off;            // This command turns the A/D module Off
    
    //  Set the inputs up to detect a Pen-down.  All are inputs except YN, which
    //      is driven to ground.  XP has it's weak pull-up enabled.  If XP is high,
    //      then no press is present.  If low, there is an active press. 
    XP = Input;              
    XN = Input;      
    YP = Input;
    YN = Output;
    XP_drive = GND;                     //Done to make sure output is at a known state
    XN_drive = GND;                     //Done to make sure output is at a known state
    YP_drive = GND;                     //Done to make sure output is at a known state
    YN_drive = GND;                     //Required to detect a press
    
    median((uint *)(&AtoD_readings),num);    // Determine the median value of the collected readings

    return(AtoD_readings[mid_index-1]);      // Return the median A/D reading as the measured value.
                                             // Have to subtract '1' because the array starts with element '0'
}


/**************************************************************
*    Function:     median(*meas,len)
*    Parameters: *meas - pointer to the data that will be evaulated
*                len   - the size of the data array
*    Return: the median value of the data
*
*    This function sorts the data from low to high and then returns
*      the middle value of the array effectiviely returning the 
*      median value. This is a common median routine found in
*      many libraries.
***************************************************************/

void median (word *meas, char len)
{
    char i,j;
    word temp;

    for(i=0; i<len; i++)
    {
        for(j=i+1; j<len; j++)
        {
            if(*(meas + i) > *(meas + j))
            {
                temp = *(meas + i);
                *(meas + i) = *(meas + j);
                *(meas + j) = temp;
            }
        }
    }
}


/**************************************************************
*    Function:     calc_xy()
*    Parameters: none
*    Return:     none
*
*    This function takes the A/D measured values for the x & y
*      coordinates and converts them to display coordinates.
*      If the coordiantes are suppose to calibrate for a mis-alignment 
*      between the display and touchscreen, then coefficients A - F have been
*      supplied by the system microprocess and stored in Flash.
*      If not, then B, C, D, and F are zero and the default translation constants
*      found in "control_constants.h" are used.
*      (Ax + By) and (Dx + Ey) are both divided by 1024 to account for the
*      1024 multiplied into the coefficients in the ts_calibrate funtion below.
*      This was done to assure we wouldn't lose significant digits doing
*      integer math.
***************************************************************/

void calc_xy (void)
{
    xtemp = ytemp = 0;
    if (xmeas > xmin)
        xtemp = xmeas - xmin;
    if (ymeas > ymin)
        ytemp = ymeas - ymin;
    xdisplay = (word)((((long)Atemp * (long)xtemp) + ((long)Btemp * (long)ytemp))>>10) + Cprime;
    ydisplay = (word)((((long)Dtemp * (long)xtemp) + ((long)Etemp * (long)ytemp))>>10) + Fprime;
}


/**************************************************************
*    Function:     ts_calibrate()
*    Parameters: minmax: 1 - read new calibration values
*    Return:     none
*
*    This function performs two separate precessor intensive calculations that do not need to be
*      done everytime the coordinates are measured.
*
*      The first is the calibration that accounts for the losses in the leads and wiring to the ts panel.  
*      It will read the max and min voltage on the two axis of the touchscreen and use these
*      values (instead of VDD and Ground) when calculating the x-y coordinates.  This is 
*      controlled by 'minmax' and is only done at power up.
*
*      The 2nd operation is to calculate a single coefficent for the translation equations in 'calc_xy'.
*      The default normalized coefficients or LCD alignment coefficents are divided by the A/D range.
*      The main benefit being not having to divide by a constant every pass thru.  The default coefficients
*      will put the measured values back to a standard value.  This can either be an A/D range -
*      1024 x 1024 for 10-bit measurement.  Or it can be to match a LCD panel size - 640 x 480 etc.
*      The normalization values are in the Aprime_default and Eprime_default variables (all other
*      defaults are 0).  These values are declared in control_constants.h.
*      For panel mis-alignment, the coefficients are received from the main processor and stored
*      in non-volatile memory at EEE_cal_data (memory location E000).
***************************************************************/

void ts_calibrate(char minmax)
{
    if(minmax)                                          //measure the min max readings of the screen
    {          
        xmax = Get_Point(read_xmax,num_readings);       //XP is not at VDD due to losses in the wire.
        xmin = Get_Point(read_xmin,num_readings);       //xmax is measured and used as the max value  
        ymax = Get_Point(read_ymax,num_readings);       //seen on the touchscreen controller.
        ymin = Get_Point(read_ymin,num_readings);       //Similar measurement taken for XN, YP, & YN
    }
    
    if(use_df_coeff)                                    //Use the default coefficents defined in control_constants.h
    {       
        Aprime = Aprime_default;
        Bprime = Bprime_default;
        Cprime = Cprime_default;
        Dprime = Dprime_default;
        Eprime = Eprime_default;
        Fprime = Fprime_default;  
    } 
    else                                                //else use the alignment calibration coefficients stored in flash
    {
        cal_ptr = (word *)(&EEE_cal_data);               
        Aprime = *cal_ptr++;
        Bprime = *cal_ptr++;
        Cprime = *cal_ptr++;
        Dprime = *cal_ptr++;
        Eprime = *cal_ptr++;
        Fprime = *cal_ptr++;    
    }
      
    //Below are processor intensive equations that only need to be done at startup or when
    //the coefficients are changed.  To stay in Integer math, the primes are multiplied by 1024.
    //This assures enough significant digits in the result to maintain resolution.    
    Atemp = (word)(((long)Aprime<<10) / (xmax - xmin));
    Btemp = (word)(((long)Bprime<<10) / (ymax - ymin));
    Dtemp = (word)(((long)Dprime<<10) / (xmax - xmin));
    Etemp = (word)(((long)Eprime<<10) / (ymax - ymin));
}


/**************************************************************
*    Function:     DelayUsec
*    Parameters: usec -- number of usec to delay. This is only
*       approximate, but the actual delay should be at least
*       as long as the requested amount.
*    Return:     none
*
***************************************************************/
void DelayUsec(word usec)
{
    volatile word i;
    word val;
    
    // one loop takes approx 2 usec (very rough)
    val = usec/2;
    for(i=0; i<val; i++)
        ;   
}